document.getElementById("resultForm").addEventListener("submit", async function(e){
  e.preventDefault();

  const admissionNo = document.getElementById("admissionNo").value.trim();
  const session = document.getElementById("session").value;
  const term = document.getElementById("term").value;

  try {
    const res = await fetch("results.json");
    const results = await res.json();

    const student = results.find(r =>
      r.admissionNo === admissionNo &&
      r.session === session &&
      r.term === term
    );

    if(student){
      // Determine average: prefer stored average if present, otherwise calculate from subjects
      let avg;
      if (student.average !== undefined && student.average !== null && student.average !== "") {
        const parsed = Number(student.average);
        avg = isNaN(parsed) ? null : Number(parsed.toFixed(2));
      }
      if (avg === null || avg === undefined) {
        const total = student.subjects.reduce((sum, sub) => sum + (Number(sub.score) || 0), 0);
        avg = Number((total / student.subjects.length).toFixed(2));
      }

      // Determine remark: prefer stored remark if present, otherwise generate from avg
      let remark = (student.remark && student.remark !== "") ? student.remark : "";
      if (!remark) {
        if (avg >= 70) remark = "Excellent";
        else if (avg >= 60) remark = "Very Good";
        else if (avg >= 50) remark = "Good";
        else if (avg >= 40) remark = "Pass";
        else remark = "Fail";
      }

      // Build subjects table
      let subjectsHTML = "";
      student.subjects.forEach(sub => {
        subjectsHTML += `<tr><td>${sub.name}</td><td>${sub.score}</td></tr>`;
      });

  document.getElementById("resultContent").innerHTML = `
        <div style="text-align:center">
          <h2>PEAK ACADEMY</h2>
          <h3>Term Result Slip</h3>
        </div>
        <h3>${student.name} (${student.admissionNo})</h3>
        <p>Class: ${student.class} | Term: ${student.term} | Session: ${student.session}</p>
        <table>
          <tr><th>Subject</th><th>Score</th></tr>
          ${subjectsHTML}
        </table>
        <p><strong>Average:</strong> ${Number(avg).toFixed(2)}</p>
        <p><strong>Remark:</strong> ${remark}</p>
      `;
      showResultCard();
    } else {
      document.getElementById("resultContent").innerHTML =
        "<p style='color:red'>❌ No result found. Check Admission No/Term/Session</p>";
      showResultCard();
    }

  } catch (err) {
    alert("⚠️ Error loading results file.");
    console.error(err);
  }
});

// Helper to reveal the result card with a slide-up animation and scroll it into view
function showResultCard() {
  const card = document.getElementById('resultCard');
  if (!card) return;

  // make visible in layout
  card.style.display = 'block';

  // ensure class is reset so animation runs every time
  card.classList.remove('show');
  // force reflow
  // eslint-disable-next-line no-unused-expressions
  card.offsetHeight;

  // add class to trigger CSS transition
  card.classList.add('show');

  // accessibility: focus card but don't jump the viewport yet
  card.setAttribute('tabindex', '-1');
  card.focus({ preventScroll: true });

  // scroll into view after a short delay so the animation is visible
  setTimeout(() => {
    card.scrollIntoView({ behavior: 'smooth', block: 'center' });
  }, 80);
}

// Print only the result card (#resultCard)
function printResult() {
  const card = document.getElementById('resultCard');
  if (!card) {
    alert('Nothing to print: result card not found.');
    return;
  }

  const newWin = window.open('', '_blank', 'width=800,height=600');
  if (!newWin) {
    alert('Unable to open print window (popup blocked).');
    return;
  }

  // Minimal styles to make the card readable when printed
  const styles = `
    <style>
      body { font-family: Inter, system-ui, -apple-system, Roboto, Arial, sans-serif; color:#101828; padding:20px }
      table { border-collapse: collapse; width:100% }
      th, td { border: 1px solid #000; padding:8px; text-align:left }
      .card, .result-card { box-shadow: none !important; border: none !important }
      .print-header { display:flex; align-items:center; gap:12px; margin-bottom:12px }
      /* Make the logo inside the result card act like a faded watermark behind the content */
      .result-card { position: relative }
      .result-card .logo { position: absolute; left: 50%; transform: translateX(-50%); top: 18px; opacity: 0.06; width: 220px; height: auto; z-index: 0 }
      #resultContent { position: relative; z-index: 1; padding-top: 12px }
    </style>
  `;

  newWin.document.write(`<!doctype html><html><head><meta charset="utf-8"><title>Print Result</title>${styles}</head><body>`);
  // small header with logo (restored) and then the card's HTML; the card's own .logo will be styled as a faded watermark
  const headerHTML = `
    <div class="print-header">
      <img src="../img/logo.png" alt="Peak Academy logo" style="width:48px;height:48px;object-fit:contain;margin-right:10px" />
      <div>
        <strong style="font-size:18px">Peak Academy</strong>
        <div style="font-size:12px;color:#667085">6 Alafia Street, Mushin, Lagos</div>
      </div>
    </div>
    <hr />
  `;

  newWin.document.write(headerHTML + card.outerHTML);
  newWin.document.write('</body></html>');
  newWin.document.close();
  newWin.focus();
  // give browser a brief moment to render, then print
  setTimeout(() => {
    newWin.print();
  }, 250);
}
